<?php
class GeminiService {
    private $apiKey;
    private $model;

    public function __construct($pdo,$model="gemini-3-pro"){
        $this->apiKey = $this->getRotatingApiKey($pdo);
        $this->model = $model;
    }

    function getRotatingApiKey($pdo){
        // کلید فعال که کمترین last_used_at دارد
        $stmt = $pdo->prepare("
            SELECT id, api_key 
            FROM api_keys 
            WHERE active = 1
            ORDER BY 
                CASE WHEN last_used_at IS NULL THEN 0 ELSE 1 END,
                last_used_at ASC
            LIMIT 1
        ");
        $stmt->execute();
        $key = $stmt->fetch(PDO::FETCH_ASSOC);

        if(!$key){
            throw new Exception("هیچ API Key فعالی وجود ندارد!");
        }

        // آپدیت زمان آخرین استفاده
        $stmt = $pdo->prepare("UPDATE api_keys SET last_used_at = NOW() WHERE id=?");
        $stmt->execute([$key["id"]]);

        return $key["api_key"];
    }

    // درخواست نرمال به generateContent -> پاسخ کامل
    public function ask($prompt){
        $url = "https://generativelanguage.googleapis.com/v1beta/models/{$this->model}:generateContent?key={$this->apiKey}";
        
        $data = ["contents"=>[["parts"=>[["text"=>$prompt]]]]];

        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => ["Content-Type: application/json"],
            CURLOPT_POSTFIELDS => json_encode($data),
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_CONNECTTIMEOUT => 5,
            CURLOPT_TIMEOUT => 120
        ]);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);

        if($response === false){
            $err = curl_error($ch);
            curl_close($ch);
            return "⚠️ خطا در تماس با Gemini: $err";
        }

        curl_close($ch);

        if($httpCode !== 200){
            return "⚠️ خطای HTTP $httpCode از Gemini";
        }

        $json = json_decode($response, true);
        if(!is_array($json)) return "⚠️ پاسخ نامعتبر از سرور Gemini.";

        $responseText = $json["candidates"][0]["content"]["parts"][0]["text"];
        $usedTokens = $this->countTokens($responseText);

        return [
            "text" => $responseText,
            "tokens" => $usedTokens
        ];
    }

    public function askWithImage($prompt, $base64Image) {

        $url = "https://generativelanguage.googleapis.com/v1beta/models/{$this->model}:generateContent?key={$this->apiKey}";

        $data = [
            "contents" => [
                [
                    "parts" => [
                        ["text" => $prompt],
                        [
                            "inline_data" => [
                                "mime_type" => "image/jpeg",
                                "data" => $base64Image
                            ]
                        ]
                    ]
                ]
            ]
        ];

        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => ["Content-Type: application/json"],
            CURLOPT_POSTFIELDS => json_encode($data),
            CURLOPT_RETURNTRANSFER => true
        ]);

        $response = curl_exec($ch);
        curl_close($ch);

        $json = json_decode($response, true);

        $responseText = $json["candidates"][0]["content"]["parts"][0]["text"];
        $usedTokens = $this->countTokens($responseText);

        return [
            "text" => $responseText,
            "tokens" => $usedTokens
        ];
    }

    public function countTokens($text){
        $length = mb_strlen($text, "UTF-8");
        return ceil($length / 5); // تقریبی: هر 5 کاراکتر ≈ 1 توکن
    }

     public function askWithContext($messages,$mode,$is_image=false,$img="")
    {
        // ساختن prompt از Context
        $prompt = "";
        foreach ($messages as $m) {
            $role = $m["role"] === "user" ? "User" : "Assistant";
            $prompt .= $role . ": " . $m["text"] . "\n";

            if($mode == 'chat_mode_talk'){
                $systemPrompt = "شما یک دستیار هوش مصنوعی حرفه‌ای و دوستانه هستید.  
وظیفه شما پاسخ‌گویی به سوالات کاربر به صورت دقیق، کامل و قابل فهم است.  
به سوالات علمی، آموزشی، عمومی و تخصصی پاسخ دهید.  
زبان پاسخ‌ها واضح، ساده و دوستانه باشد تا حتی کاربران تازه‌کار هم راحت متوجه شوند.  
در صورت نیاز، توضیح اضافی یا مثال عملی ارائه کنید تا کاربر بهتر درک کند.  
لحن پاسخ‌ها محترمانه و صمیمی باشد و حس یک همراه و مشاور شخصی را منتقل کند.  
اگر سوال کاربر مبهم یا ناقص بود، ابتدا از او سوال تکمیلی بپرسید تا پاسخ دقیق بدهید.
";
            }elseif($mode == 'chat_mode_code'){

                $systemPrompt = "
                You are a professional coding assistant.
                Return your output ONLY in the following structure:

                ===CAPTION===
                (Write a short description for the user here)
                ===ENDCAPTION===

                ===CODE===
                (Write only the final executable code here. Do NOT add explanations.)
                ===ENDCODE===

                Do NOT add anything outside these blocks.";

            }elseif($mode == 'chat_mode_content'){
                $systemPrompt = "شما یک دستیار حرفه‌ای نویسنده هستید. متن‌های با کیفیت، خلاقانه و مطابق با سبک و لحن کاربر تولید کنید.  
اگر کاربر موضوعی ارائه کرد، مقاله، کپشن، ایمیل یا متن مرتبط و کامل بنویسید.  
متن باید روان، جذاب و قابل فهم باشد و از غلط‌های املایی و نگارشی جلوگیری شود.  
در صورت نیاز، چند نمونه یا بخش‌بندی برای راحتی خواندن ارائه کنید.  
همیشه لحن کاربر را رعایت کنید و پیشنهادهای مفید برای بهبود متن بدهید.
";
            }elseif($mode == 'chat_mode_translate'){
                $systemPrompt = "شما یک مترجم حرفه‌ای هستید که می‌تواند متون را به هر زبان با دقت و حفظ لحن و مفهوم ترجمه کند.  
متن کاربر را ترجمه کنید و زبان مقصد را رعایت کنید.  
لحن متن، اصطلاحات و سبک نوشتاری را حفظ کنید.  
اگر متن شامل جملات پیچیده یا تخصصی است، ترجمه‌ای دقیق و قابل فهم ارائه دهید.  
در صورت نیاز، معادل‌های جایگزین برای اصطلاحات ارائه دهید تا کاربر بتواند بهترین انتخاب را داشته باشد.
";
            }elseif($mode == 'chat_mode_image'){
                $systemPrompt = "شما یک طراح و تولیدکننده تصاویر هستید. بر اساس توضیحات کاربر، تصویر یا طرح گرافیکی با کیفیت و دقیق ایجاد کنید.  
ویژگی‌ها، رنگ‌ها، سبک و جزئیات ارائه شده توسط کاربر را رعایت کنید.  
اگر توضیحات مبهم بود، چند پیشنهاد واضح و قابل انتخاب ارائه دهید.  
خروجی باید قابل ارائه و آمادهٔ استفاده باشد و مطابق با سبک مدرن و جذاب باشد.  
در صورت امکان، گزینه‌های خلاقانه برای بهتر شدن تصویر ارائه دهید.
";
            }elseif($mode == 'chat_mode_analysis'){
                $systemPrompt = "شما یک تحلیلگر داده حرفه‌ای هستید. داده‌ها و اطلاعات ارائه شده توسط کاربر را پردازش و تحلیل کنید.  
نتایج دقیق، واضح و قابل فهم ارائه کنید.  
اگر داده‌ها قابل گرافیک یا نمودار هستند، توضیح دهید که چگونه می‌توان آن‌ها را نمایش داد.  
توصیه‌ها و نکات عملی بر اساس داده‌ها ارائه کنید.  
تحلیل باید به زبان ساده، حرفه‌ای و قابل استفاده برای تصمیم‌گیری باشد.
";}
            $prompt .= "system" . ": " . $systemPrompt . "\n";

        }

        if($is_image){
            return $this->askWithImage($prompt,$img);
        }else{
            return $this->ask($prompt);
        }
        // ارسال prompt به متد اصلی ask
        
    }
    


}
